<?php
require_once '../includes/config.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../classes/Transaction.php';
require_once '../classes/User.php';
require_once '../includes/mailer.php';

checkAuthentication();

$user = new User();
$currentUser = $user->getUserById($_SESSION['user_id']);
$transaction = new Transaction();

$success = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['amount'], $_POST['method'])) {
    $amount = (float)$_POST['amount'];
    $method = trim($_POST['method']);
    
    // Validate
    if ($amount <= 0) {
        $error = 'Invalid amount';
    } elseif ($amount > $currentUser['balance']) {
        $error = 'Insufficient balance';
    } else {
        // Create withdrawal request
        $transactionId = $transaction->create([
            'user_id' => $_SESSION['user_id'],
            'type' => 'withdrawal',
            'amount' => $amount,
            'method' => $method,
            'status' => 'pending'
        ]);
        
        // Send notification to admin
        sendEmail(
            'admin@yourdomain.com',
            'New Withdrawal Request',
            'admin_withdrawal_notification',
            [
                'user_name' => $currentUser['name'],
                'user_email' => $currentUser['email'],
                'amount' => $amount,
                'method' => $method
            ]
        );
        
        // Send confirmation to user
        sendEmail(
            $currentUser['email'],
            'Withdrawal Request Received',
            'withdrawal_requested',
            [
                'name' => $currentUser['name'],
                'amount' => $amount,
                'method' => $method
            ]
        );
        
        $success = 'Withdrawal request submitted successfully';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Withdraw Funds | InvestPro</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        .gradient-bg {
            background: linear-gradient(135deg, #1A237E 0%, #009688 100%);
        }
    </style>
</head>
<body class="bg-gray-50">
    <?php include '../includes/header.php'; ?>
    
    <div class="container mx-auto px-4 py-8">
        <div class="max-w-2xl mx-auto">
            <div class="bg-white rounded-xl shadow-md p-6 mb-6">
                <h2 class="text-2xl font-bold mb-6">Withdraw Funds</h2>
                
                <?php if ($error): ?>
                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                        <?php echo $error; ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($success): ?>
                    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                        <?php echo $success; ?>
                    </div>
                <?php endif; ?>
                
                <div class="mb-6 p-4 border border-gray-200 rounded-lg bg-gray-50">
                    <div class="flex justify-between items-center">
                        <span class="text-gray-600">Available Balance:</span>
                        <span class="text-xl font-bold">$<?php echo number_format($currentUser['balance'], 2); ?></span>
                    </div>
                </div>
                
                <form method="POST">
                    <div class="mb-4">
                        <label for="amount" class="block text-gray-700 font-medium mb-2">Amount</label>
                        <input type="number" id="amount" name="amount" step="0.01" min="10" max="<?php echo $currentUser['balance']; ?>" required 
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" 
                            placeholder="Minimum $10">
                    </div>
                    
                    <div class="mb-6">
                        <label for="method" class="block text-gray-700 font-medium mb-2">Withdrawal Method</label>
                        <select id="method" name="method" required class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="">Select Method</option>
                            <option value="Bank Transfer">Bank Transfer</option>
                            <option value="Bitcoin">Bitcoin</option>
                            <option value="Ethereum">Ethereum</option>
                            <option value="USDT">USDT (TRC20/ERC20)</option>
                            <option value="PayPal">PayPal</option>
                        </select>
                    </div>
                    
                    <button type="submit" class="w-full gradient-bg text-white py-2 px-4 rounded-lg font-medium hover:opacity-90 transition">Request Withdrawal</button>
                </form>
            </div>
            
            <div class="bg-white rounded-xl shadow-md p-6">
                <h3 class="text-xl font-bold mb-4">Withdrawal History</h3>
                
                <?php
                $withdrawals = $transaction->getUserTransactions($_SESSION['user_id'], 'withdrawal');
                if (!empty($withdrawals)):
                ?>
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Method</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($withdrawals as $withdrawal): ?>
                                <tr>
                                    <td class="px-6 py-4 whitespace-nowrap"><?php echo date('M d, Y', strtotime($withdrawal['created_at'])); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap">$<?php echo number_format($withdrawal['amount'], 2); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap"><?php echo htmlspecialchars($withdrawal['method']); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full 
                                            <?php echo $withdrawal['status'] === 'approved' ? 'bg-green-100 text-green-800' : 
                                                 ($withdrawal['status'] === 'rejected' ? 'bg-red-100 text-red-800' : 'bg-yellow-100 text-yellow-800'); ?>">
                                            <?php echo ucfirst($withdrawal['status']); ?>
                                        </span>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-8">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-12 w-12 mx-auto text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                        <h4 class="mt-2 text-lg font-medium text-gray-700">No withdrawal history</h4>
                        <p class="mt-1 text-gray-500">Your withdrawal requests will appear here</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <?php include '../includes/footer.php'; ?>
</body>
</html>