<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../classes/User.php';

checkAuthentication();

$user = new User();
$currentUser = $user->getUserById($_SESSION['user_id']);

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Handle profile update
    if (isset($_POST['update_profile'])) {
        $name = trim($_POST['name']);
        $email = trim($_POST['email']);
        
        if (empty($name)) {
            $error = 'Name is required';
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = 'Valid email is required';
        } else {
            if ($user->updateProfile($_SESSION['user_id'], $name, $email)) {
                $_SESSION['user_name'] = $name;
                $success = 'Profile updated successfully';
                $currentUser = $user->getUserById($_SESSION['user_id']); // Refresh data
            } else {
                $error = 'Failed to update profile';
            }
        }
    }
    
    // Handle password change
    if (isset($_POST['change_password'])) {
        $currentPassword = $_POST['current_password'];
        $newPassword = $_POST['new_password'];
        $confirmPassword = $_POST['confirm_password'];
        
        if (!password_verify($currentPassword, $currentUser['password'])) {
            $error = 'Current password is incorrect';
        } elseif (strlen($newPassword) < 8) {
            $error = 'New password must be at least 8 characters';
        } elseif ($newPassword !== $confirmPassword) {
            $error = 'Passwords do not match';
        } else {
            if ($user->changePassword($_SESSION['user_id'], $newPassword)) {
                $success = 'Password changed successfully';
            } else {
                $error = 'Failed to change password';
            }
        }
    }
}

include __DIR__ . '/../includes/header.php';
?>

<div class="container mx-auto px-4 py-8">
    <div class="max-w-4xl mx-auto">
        <h1 class="text-3xl font-bold mb-6">Your Profile</h1>
        
        <?php if ($error): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="grid grid-cols-1 md:grid-cols-3 gap-8">
            <!-- Profile Summary -->
            <div class="md:col-span-1">
                <div class="bg-white rounded-xl shadow-md p-6 text-center">
                    <div class="h-24 w-24 mx-auto rounded-full bg-teal-100 flex items-center justify-center text-teal-600 text-4xl font-bold mb-4">
                        <?php echo strtoupper(substr($currentUser['name'], 0, 1)); ?>
                    </div>
                    <h2 class="text-xl font-bold"><?php echo htmlspecialchars($currentUser['name']); ?></h2>
                    <p class="text-gray-600"><?php echo htmlspecialchars($currentUser['email']); ?></p>
                    
                    <div class="mt-6 pt-6 border-t border-gray-200">
                        <div class="flex justify-between mb-2">
                            <span class="text-gray-600">Member Since:</span>
                            <span class="font-medium"><?php echo date('M Y', strtotime($currentUser['created_at'])); ?></span>
                        </div>
                        <div class="flex justify-between">
                            <span class="text-gray-600">Email Verified:</span>
                            <span class="font-medium">
                                <?php echo $currentUser['email_verified_at'] ? 'Yes' : 'No'; ?>
                            </span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Profile Details -->
            <div class="md:col-span-2">
                <div class="bg-white rounded-xl shadow-md p-6 mb-6">
                    <h2 class="text-xl font-bold mb-4">Profile Information</h2>
                    
                    <form method="POST">
                        <input type="hidden" name="update_profile" value="1">
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                            <div>
                                <label for="name" class="block text-gray-700 font-medium mb-2">Full Name</label>
                                <input type="text" id="name" name="name" required
                                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                                    value="<?php echo htmlspecialchars($currentUser['name']); ?>">
                            </div>
                            <div>
                                <label for="email" class="block text-gray-700 font-medium mb-2">Email Address</label>
                                <input type="email" id="email" name="email" required
                                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                                    value="<?php echo htmlspecialchars($currentUser['email']); ?>">
                            </div>
                        </div>
                        
                        <button type="submit" 
                            class="gradient-bg text-white py-2 px-6 rounded-lg font-medium hover:opacity-90 transition">
                            Update Profile
                        </button>
                    </form>
                </div>
                
                <div class="bg-white rounded-xl shadow-md p-6">
                    <h2 class="text-xl font-bold mb-4">Change Password</h2>
                    
                    <form method="POST">
                        <input type="hidden" name="change_password" value="1">
                        
                        <div class="mb-4">
                            <label for="current_password" class="block text-gray-700 font-medium mb-2">Current Password</label>
                            <input type="password" id="current_password" name="current_password" required
                                class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                            <div>
                                <label for="new_password" class="block text-gray-700 font-medium mb-2">New Password</label>
                                <input type="password" id="new_password" name="new_password" required
                                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label for="confirm_password" class="block text-gray-700 font-medium mb-2">Confirm Password</label>
                                <input type="password" id="confirm_password" name="confirm_password" required
                                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                        </div>
                        
                        <button type="submit" 
                            class="gradient-bg text-white py-2 px-6 rounded-lg font-medium hover:opacity-90 transition">
                            Change Password
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include __DIR__ . '/../includes/footer.php'; ?>