<?php
class Investment {
    private $db;
    
    public function __construct() {
        $this->db = getDBConnection();
    }
    
    public function getAvailablePlans() {
        $stmt = $this->db->prepare("SELECT * FROM investment_plans WHERE is_active = 1");
        $stmt->execute();
        $result = $stmt->get_result();
        
        $plans = [];
        while ($row = $result->fetch_assoc()) {
            $plans[] = $row;
        }
        
        return $plans;
    }
    
    public function getActiveInvestments($userId) {
        $stmt = $this->db->prepare("
            SELECT ui.*, ip.name as plan_name 
            FROM user_investments ui
            JOIN investment_plans ip ON ui.plan_id = ip.id
            WHERE ui.user_id = ? AND ui.is_active = 1
        ");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $investments = [];
        while ($row = $result->fetch_assoc()) {
            $investments[] = $row;
        }
        
        return $investments;
    }
    
    public function createInvestment($userId, $planId, $amount) {
        // Get plan details first
        $plan = $this->getPlanById($planId);
        if (!$plan) {
            throw new Exception("Invalid investment plan");
        }
        
        // Validate amount against plan limits
        if ($amount < $plan['min_amount'] || ($plan['max_amount'] && $amount > $plan['max_amount'])) {
            throw new Exception("Amount does not meet plan requirements");
        }
        
        // Calculate end date based on duration
        $endDate = date('Y-m-d H:i:s', strtotime("+{$plan['duration_days']} days"));
        
        // Calculate first payout date based on interval type
        switch ($plan['interval_type']) {
            case 'daily':
                $nextPayout = date('Y-m-d H:i:s', strtotime('+1 day'));
                break;
            case 'weekly':
                $nextPayout = date('Y-m-d H:i:s', strtotime('+1 week'));
                break;
            case 'monthly':
                $nextPayout = date('Y-m-d H:i:s', strtotime('+1 month'));
                break;
            default: // custom
                $nextPayout = $endDate;
        }
        
        $stmt = $this->db->prepare("
            INSERT INTO user_investments 
            (user_id, plan_id, amount, start_date, end_date, next_payout, is_active) 
            VALUES (?, ?, ?, NOW(), ?, ?, 1)
        ");
        $stmt->bind_param("iidss", $userId, $planId, $amount, $endDate, $nextPayout);
        
        if (!$stmt->execute()) {
            throw new Exception("Failed to create investment: " . $stmt->error);
        }
        
        return $this->db->insert_id;
    }
    
    public function getPlanById($planId) {
        $stmt = $this->db->prepare("SELECT * FROM investment_plans WHERE id = ?");
        $stmt->bind_param("i", $planId);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }
    
    public function calculateTotalROI($userId) {
        $stmt = $this->db->prepare("
            SELECT SUM(total_earned) as total 
            FROM user_investments 
            WHERE user_id = ?
        ");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        
        return $row['total'] ? $row['total'] : 0;
    }
    
    public function recordPayout($investmentId, $amount) {
        $stmt = $this->db->prepare("
            INSERT INTO payout_logs (investment_id, amount) 
            VALUES (?, ?)
        ");
        $stmt->bind_param("id", $investmentId, $amount);
        return $stmt->execute();
    }
}
?>

<div class="container mx-auto px-4 py-8">
    <div class="max-w-4xl mx-auto">
        <h1 class="text-3xl font-bold mb-6">Invest Funds</h1>
        
        <?php if ($error): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
            <!-- Investment Form -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-xl font-bold mb-4">New Investment</h2>
                
                <div class="mb-6 p-4 border border-gray-200 rounded-lg bg-gray-50">
                    <div class="flex justify-between items-center">
                        <span class="text-gray-600">Available Balance:</span>
                        <span class="text-xl font-bold">$<?php echo number_format($currentUser['balance'], 2); ?></span>
                    </div>
                </div>
                
                <form method="POST">
                    <div class="mb-4">
                        <label for="plan_id" class="block text-gray-700 font-medium mb-2">Select Plan</label>
                        <select id="plan_id" name="plan_id" required 
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="">Choose an investment plan</option>
                            <?php foreach ($plans as $plan): ?>
                            <option value="<?php echo $plan['id']; ?>" 
                                data-min="<?php echo $plan['min_amount']; ?>"
                                data-max="<?php echo $plan['max_amount']; ?>">
                                <?php echo htmlspecialchars($plan['name']); ?> - 
                                <?php echo $plan['roi_percentage']; ?>% ROI - 
                                Min: $<?php echo number_format($plan['min_amount'], 2); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-4">
                        <label for="amount" class="block text-gray-700 font-medium mb-2">Investment Amount ($)</label>
                        <input type="number" id="amount" name="amount" step="0.01" min="0" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <p class="text-sm text-gray-500 mt-1" id="amount-hint">Select a plan to see amount limits</p>
                    </div>
                    
                    <button type="submit" 
                        class="w-full gradient-bg text-white py-2 px-4 rounded-lg font-medium hover:opacity-90 transition">
                        Invest Now
                    </button>
                </form>
            </div>
            
            <!-- Plan Details -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-xl font-bold mb-4">Investment Plans</h2>
                
                <div class="space-y-4">
                    <?php foreach ($plans as $plan): ?>
                    <div class="border border-gray-200 rounded-lg p-4">
                        <h3 class="font-bold text-lg"><?php echo htmlspecialchars($plan['name']); ?></h3>
                        <div class="grid grid-cols-2 gap-2 mt-2">
                            <div>
                                <span class="text-gray-600">ROI:</span>
                                <span class="font-medium"><?php echo $plan['roi_percentage']; ?>%</span>
                            </div>
                            <div>
                                <span class="text-gray-600">Duration:</span>
                                <span class="font-medium"><?php echo $plan['duration_days']; ?> days</span>
                            </div>
                            <div>
                                <span class="text-gray-600">Min Amount:</span>
                                <span class="font-medium">$<?php echo number_format($plan['min_amount'], 2); ?></span>
                            </div>
                            <div>
                                <span class="text-gray-600">Max Amount:</span>
                                <span class="font-medium"><?php echo $plan['max_amount'] ? '$'.number_format($plan['max_amount'], 2) : 'Unlimited'; ?></span>
                            </div>
                        </div>
                        <p class="mt-2 text-gray-700"><?php echo htmlspecialchars($plan['description']); ?></p>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
        <!-- Active Investments -->
        <div class="mt-8 bg-white rounded-xl shadow-md p-6">
            <h2 class="text-xl font-bold mb-4">Your Active Investments</h2>
            
            <?php $activeInvestments = $investment->getActiveInvestments($_SESSION['user_id']); ?>
            <?php if (!empty($activeInvestments)): ?>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Plan</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ROI Earned</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Start Date</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">End Date</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($activeInvestments as $inv): ?>
                            <tr>
                                <td class="px-6 py-4 whitespace-nowrap"><?php echo htmlspecialchars($inv['plan_name']); ?></td>
                                <td class="px-6 py-4 whitespace-nowrap">$<?php echo number_format($inv['amount'], 2); ?></td>
                                <td class="px-6 py-4 whitespace-nowrap">$<?php echo number_format($inv['total_earned'], 2); ?></td>
                                <td class="px-6 py-4 whitespace-nowrap"><?php echo date('M d, Y', strtotime($inv['start_date'])); ?></td>
                                <td class="px-6 py-4 whitespace-nowrap"><?php echo date('M d, Y', strtotime($inv['end_date'])); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="text-center py-8">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-12 w-12 mx-auto text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                    </svg>
                    <h4 class="mt-2 text-lg font-medium text-gray-700">No active investments</h4>
                    <p class="mt-1 text-gray-500">Your active investments will appear here</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
document.getElementById('plan_id').addEventListener('change', function() {
    const selectedOption = this.options[this.selectedIndex];
    const minAmount = selectedOption.dataset.min;
    const maxAmount = selectedOption.dataset.max;
    const amountInput = document.getElementById('amount');
    
    amountInput.min = minAmount;
    if (maxAmount) {
        amountInput.max = maxAmount;
    }
    
    const hint = document.getElementById('amount-hint');
    if (minAmount) {
        hint.textContent = `Minimum: $${parseFloat(minAmount).toFixed(2)}` + 
                         (maxAmount ? `, Maximum: $${parseFloat(maxAmount).toFixed(2)}` : '');
    } else {
        hint.textContent = 'Select a plan to see amount limits';
    }
});
</script>

<?php include __DIR__ . '/../includes/footer.php'; ?>