<?php
require_once '../includes/config.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../classes/User.php';
require_once '../classes/Investment.php';
require_once '../classes/Transaction.php';

// Check authentication
checkAuthentication();

// Get current user
$user = new User();
$currentUser = $user->getUserById($_SESSION['user_id']);

// Get user investments
$investment = new Investment();
$activeInvestments = $investment->getActiveInvestments($_SESSION['user_id']);

// Get recent transactions
$transaction = new Transaction();
$recentTransactions = $transaction->getRecentTransactions($_SESSION['user_id'], 5);

// Calculate total ROI
$totalROI = $investment->calculateTotalROI($_SESSION['user_id']);

// Include header
include '../includes/header.php';
?>

<div class="container mx-auto px-4 py-8">
    <!-- Welcome Banner -->
    <div class="gradient-bg text-white rounded-xl p-6 mb-8 shadow-lg">
        <h1 class="text-3xl font-bold">Welcome back, <?php echo htmlspecialchars($currentUser['name']); ?></h1>
        <p class="opacity-90">Track your investments and grow your portfolio</p>
    </div>

    <!-- Stats Cards -->
    <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
        <!-- Balance Card -->
        <div class="bg-white rounded-xl shadow-md p-6 card-hover">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-blue-100 text-blue-600 mr-4">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                    </svg>
                </div>
                <div>
                    <p class="text-gray-500">Available Balance</p>
                    <h3 class="text-2xl font-bold">$<?php echo number_format($currentUser['balance'], 2); ?></h3>
                </div>
            </div>
        </div>

        <!-- Active Investments Card -->
        <div class="bg-white rounded-xl shadow-md p-6 card-hover">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-teal-100 text-teal-600 mr-4">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                    </svg>
                </div>
                <div>
                    <p class="text-gray-500">Active Investments</p>
                    <h3 class="text-2xl font-bold"><?php echo count($activeInvestments); ?></h3>
                </div>
            </div>
        </div>

        <!-- Total ROI Card -->
        <div class="bg-white rounded-xl shadow-md p-6 card-hover">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-purple-100 text-purple-600 mr-4">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
                    </svg>
                </div>
                <div>
                    <p class="text-gray-500">Total ROI Earned</p>
                    <h3 class="text-2xl font-bold">$<?php echo number_format($totalROI, 2); ?></h3>
                </div>
            </div>
        </div>
    </div>

    <!-- Charts Section -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
        <!-- ROI Chart -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <h3 class="text-xl font-bold mb-4">ROI Performance</h3>
            <canvas id="roiChart" height="300"></canvas>
        </div>

        <!-- Investment Distribution -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <h3 class="text-xl font-bold mb-4">Investment Distribution</h3>
            <canvas id="investmentChart" height="300"></canvas>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
        <!-- Deposit Card -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <h3 class="text-xl font-bold mb-4">Deposit Funds</h3>
            <p class="text-gray-600 mb-4">Add funds to your account to start investing</p>
            <a href="deposit.php" class="inline-block gradient-bg text-white px-6 py-2 rounded-lg font-medium hover:opacity-90 transition">Deposit Now</a>
        </div>

        <!-- Invest Card -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <h3 class="text-xl font-bold mb-4">Start Investing</h3>
            <p class="text-gray-600 mb-4">Choose from our profitable investment plans</p>
            <a href="invest.php" class="inline-block gradient-bg text-white px-6 py-2 rounded-lg font-medium hover:opacity-90 transition">Invest Now</a>
        </div>
    </div>

    <!-- Active Investments -->
    <div class="bg-white rounded-xl shadow-md p-6 mb-8">
        <div class="flex justify-between items-center mb-6">
            <h3 class="text-xl font-bold">Active Investments</h3>
            <a href="invest.php" class="text-blue-600 hover:underline">View All</a>
        </div>
        
        <?php if (count($activeInvestments) > 0): ?>
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Plan</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ROI</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Start Date</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">End Date</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php foreach ($activeInvestments as $investment): ?>
                        <tr>
                            <td class="px-6 py-4 whitespace-nowrap"><?php echo htmlspecialchars($investment['plan_name']); ?></td>
                            <td class="px-6 py-4 whitespace-nowrap">$<?php echo number_format($investment['amount'], 2); ?></td>
                            <td class="px-6 py-4 whitespace-nowrap">$<?php echo number_format($investment['total_earned'], 2); ?></td>
                            <td class="px-6 py-4 whitespace-nowrap"><?php echo date('M d, Y', strtotime($investment['start_date'])); ?></td>
                            <td class="px-6 py-4 whitespace-nowrap"><?php echo date('M d, Y', strtotime($investment['end_date'])); ?></td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">Active</span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="text-center py-8">
                <svg xmlns="http://www.w3.org/2000/svg" class="h-12 w-12 mx-auto text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                </svg>
                <h4 class="mt-2 text-lg font-medium text-gray-700">No active investments</h4>
                <p class="mt-1 text-gray-500">Start investing to see your active plans here</p>
                <a href="invest.php" class="mt-4 inline-block gradient-bg text-white px-6 py-2 rounded-lg font-medium hover:opacity-90 transition">Invest Now</a>
            </div>
        <?php endif; ?>
    </div>

    <!-- Recent Transactions -->
    <div class="bg-white rounded-xl shadow-md p-6">
        <div class="flex justify-between items-center mb-6">
            <h3 class="text-xl font-bold">Recent Transactions</h3>
            <a href="transactions.php" class="text-blue-600 hover:underline">View All</a>
        </div>
        
        <?php if (count($recentTransactions) > 0): ?>
            <div class="space-y-4">
                <?php foreach ($recentTransactions as $transaction): ?>
                <div class="flex items-center justify-between p-4 border border-gray-100 rounded-lg hover:bg-gray-50">
                    <div class="flex items-center">
                        <?php if ($transaction['type'] === 'deposit'): ?>
                            <div class="p-3 rounded-full bg-blue-100 text-blue-600 mr-4">
                                <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                            </div>
                        <?php elseif ($transaction['type'] === 'withdrawal'): ?>
                            <div class="p-3 rounded-full bg-red-100 text-red-600 mr-4">
                                <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 14l-7 7m0 0l-7-7m7 7V3" />
                                </svg>
                            </div>
                        <?php else: ?>
                            <div class="p-3 rounded-full bg-green-100 text-green-600 mr-4">
                                <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
                                </svg>
                            </div>
                        <?php endif; ?>
                        <div>
                            <h4 class="font-medium"><?php echo ucfirst($transaction['type']); ?></h4>
                            <p class="text-sm text-gray-500"><?php echo date('M d, Y h:i A', strtotime($transaction['created_at'])); ?></p>
                        </div>
                    </div>
                    <div class="text-right">
                        <p class="font-medium <?php echo $transaction['type'] === 'withdrawal' ? 'text-red-600' : 'text-green-600'; ?>">
                            <?php echo $transaction['type'] === 'withdrawal' ? '-' : '+'; ?>$<?php echo number_format($transaction['amount'], 2); ?>
                        </p>
                        <p class="text-sm text-gray-500"><?php echo ucfirst($transaction['status']); ?></p>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="text-center py-8">
                <svg xmlns="http://www.w3.org/2000/svg" class="h-12 w-12 mx-auto text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 14l6-6m-5.5.5h.01m4.99 5h.01M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16l3.5-2 3.5 2 3.5-2 3.5 2z" />
                </svg>
                <h4 class="mt-2 text-lg font-medium text-gray-700">No transactions yet</h4>
                <p class="mt-1 text-gray-500">Your transactions will appear here</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<script src="../assets/js/dashboard.js"></script>
<script>
    // Pass PHP data to JavaScript
    const chartData = {
        roiPerformance: <?php echo json_encode(getROIPerformanceData($_SESSION['user_id'])); ?>,
        investmentDistribution: <?php echo json_encode(getInvestmentDistributionData($_SESSION['user_id'])); ?>
    };
</script>

<?php
// Include footer
include '../includes/footer.php';
?>