<?php
require_once __DIR__.'/../includes/config.php';
require_once __DIR__.'/../includes/db.php';
require_once __DIR__.'/../classes/Investment.php';
require_once __DIR__.'/../classes/Transaction.php';
require_once __DIR__.'/../classes/User.php';
require_once __DIR__.'/../includes/mailer.php';

// Initialize classes
$investment = new Investment();
$transaction = new Transaction();
$user = new User();

// Get all active investments that are due for payout
$dueInvestments = $investment->getDueInvestments();

foreach ($dueInvestments as $investmentRecord) {
    try {
        // Calculate ROI amount
        $roiAmount = $investmentRecord['amount'] * ($investmentRecord['roi_percentage'] / 100);
        
        // Start transaction
        $db = getDBConnection();
        $db->autocommit(false);
        
        // 1. Credit user's wallet
        $user->updateBalance($investmentRecord['user_id'], $roiAmount);
        
        // 2. Record ROI transaction
        $transaction->create([
            'user_id' => $investmentRecord['user_id'],
            'type' => 'roi',
            'amount' => $roiAmount,
            'status' => 'approved'
        ]);
        
        // 3. Update investment record
        $investment->updateInvestmentAfterPayout(
            $investmentRecord['id'],
            $roiAmount,
            $investment->calculateNextPayoutDate($investmentRecord)
        );
        
        // 4. Record payout log
        $investment->recordPayout(
            $investmentRecord['id'],
            $roiAmount
        );
        
        // Check if investment has completed
        if (strtotime($investmentRecord['end_date']) <= time()) {
            $investment->markAsCompleted($investmentRecord['id']);
            
            // Send plan expiry notification
            $userData = $user->getUserById($investmentRecord['user_id']);
            sendEmail(
                $userData['email'],
                'Investment Plan Completed',
                'investment_expired',
                [
                    'name' => $userData['name'],
                    'plan_name' => $investmentRecord['plan_name'],
                    'amount' => $investmentRecord['amount'],
                    'total_earned' => $investmentRecord['total_earned'] + $roiAmount
                ]
            );
        }
        
        // Commit transaction
        $db->commit();
        
        // Send ROI credit notification
        $userData = $user->getUserById($investmentRecord['user_id']);
        sendEmail(
            $userData['email'],
            'ROI Credited to Your Account',
            'roi_credited',
            [
                'name' => $userData['name'],
                'amount' => $roiAmount,
                'plan_name' => $investmentRecord['plan_name'],
                'balance' => $userData['balance'] + $roiAmount
            ]
        );
        
    } catch (Exception $e) {
        // Rollback on error
        $db->rollback();
        error_log("Investment processing failed: " . $e->getMessage());
    }
}

echo "Processed " . count($dueInvestments) . " investments\n";
?>