<?php
// classes/Investment.php

require_once __DIR__ . '/../config/db.php'; // adjust path to your DB connection file if needed

class Investment {
    private $db;

    public function __construct() {
        $this->db = getDBConnection();
    }

    // Get active investments for a user
    public function getActiveInvestments($userId) {
        $stmt = $this->db->prepare("
            SELECT ui.*, ip.name as plan_name 
            FROM user_investments ui
            JOIN investment_plans ip ON ui.plan_id = ip.id
            WHERE ui.user_id = ? AND ui.is_active = 1
        ");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();

        $investments = [];
        while ($row = $result->fetch_assoc()) {
            $investments[] = $row;
        }

        return $investments;
    }

    // Calculate total ROI for a user
    public function calculateTotalROI($userId) {
        $stmt = $this->db->prepare("
            SELECT SUM(total_earned) as total 
            FROM user_investments 
            WHERE user_id = ?
        ");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();

        return $row['total'] ? $row['total'] : 0;
    }

    // Create a new investment
    public function createInvestment($userId, $planId, $amount) {
        // Step 1: Get plan duration and interval
        $stmt = $this->db->prepare("SELECT duration_days, interval_type FROM investment_plans WHERE id = ?");
        $stmt->bind_param("i", $planId);
        $stmt->execute();
        $result = $stmt->get_result();
        $plan = $result->fetch_assoc();

        if (!$plan) return false;

        // Step 2: Calculate end date and next payout
        $startDate = date("Y-m-d H:i:s");
        $endDate = date("Y-m-d H:i:s", strtotime("+{$plan['duration_days']} days"));

        // Determine payout interval in days
        switch ($plan['interval_type']) {
            case 'daily':
                $intervalDays = 1;
                break;
            case 'weekly':
                $intervalDays = 7;
                break;
            case 'monthly':
                $intervalDays = 30;
                break;
            default:
                $intervalDays = $plan['duration_days'];
        }

        $nextPayout = date("Y-m-d H:i:s", strtotime("+{$intervalDays} days"));

        // Step 3: Insert the investment
        $stmt = $this->db->prepare("
            INSERT INTO user_investments 
            (user_id, plan_id, amount, start_date, end_date, next_payout) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->bind_param("iidsss", $userId, $planId, $amount, $startDate, $endDate, $nextPayout);
        return $stmt->execute();
    }

    // Optionally: Deactivate expired investments
    public function deactivateExpiredInvestments() {
        $stmt = $this->db->prepare("
            UPDATE user_investments 
            SET is_active = 0 
            WHERE end_date <= NOW() AND is_active = 1
        ");
        return $stmt->execute();
    }

    // Optionally: Get all investments of a user
    public function getUserInvestmentHistory($userId) {
        $stmt = $this->db->prepare("
            SELECT ui.*, ip.name as plan_name 
            FROM user_investments ui
            JOIN investment_plans ip ON ui.plan_id = ip.id
            WHERE ui.user_id = ?
            ORDER BY ui.start_date DESC
        ");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();

        $history = [];
        while ($row = $result->fetch_assoc()) {
            $history[] = $row;
        }

        return $history;
    }

    // Optionally: Update ROI for an investment
    public function updateROI($investmentId, $amount) {
        $stmt = $this->db->prepare("
            UPDATE user_investments 
            SET total_earned = total_earned + ?, next_payout = DATE_ADD(next_payout, INTERVAL 1 DAY)
            WHERE id = ?
        ");
        $stmt->bind_param("di", $amount, $investmentId);
        return $stmt->execute();
    }
}
